﻿using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using NVCC.Models;

namespace NVCC.WebUI.Controllers
{
    [NvccAuthorize]
    public class RoiController : Controller
    {
        private readonly IRoiService _roiService;
        private IUserService _userService;
        private readonly IAccessiblePdfService _accessiblePdfService;

        private static readonly string MaxNumberOfAttachedPdfsErrorMessage = string.Format("required");

        public RoiController(IRoiService roiService,
                             IUserService userService,
                             IAccessiblePdfService accessiblePdfService)
        {
            _roiService = roiService;
            _userService = userService;
            _accessiblePdfService = accessiblePdfService;
        }

        [HttpGet]
        public ActionResult Index()
        {
            return View(new RoiSearhViewModel());
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Index(RoiSearhViewModel model)
        {
            ViewBag.UserIsRoiAdmin = _userService.CheckRoiAdminAuthorization() > 0;
            if (ModelState.IsValid)
            {
                model.SearchResults = _roiService.RoiSearch(model.LastFourOfSsn, model.LastName, model.FirstFiveOfSsn, model.FirstName, model.StartDate, model.EndDate);
            }
            return View(model);
        }

        [HttpPost]
        [RoiAdminAuthorize]
        [ValidateAntiForgeryToken]
        public ActionResult Admin(RoiAdminSearchViewModel model)
        {
                //username is optional
                if (model.selectedUserName == "0")
                {
                    model.selectedUserName = null;
                }

            int isIncompleteAODChecked = 0;

                if (model.incompleteAODChecked)
                {
                    isIncompleteAODChecked = 1;
                }
               
                            
                model.roiAdminSearchList = _roiService.GetRoiAdminSearchResults(model.facilityId, model.selectedVisnId, model.StartDate, model.EndDate, model.selectedUserName, isIncompleteAODChecked);
                model.visnList = _userService.GetVisn().ToList();

                //load facilities
                int selectedVisnId = model.selectedVisnId;
                model.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == selectedVisnId).ToList();

                //load username list
                int selectedfacilityId = model.facilityId;
                model.UserList = _roiService.GetUsersByFacilityId(selectedfacilityId);

                const string roiAdminSessionKey = "RoiAdminSearch";
                if (HttpContextManager.Current.Session != null)
                {
                    HttpContextManager.Current.Session[roiAdminSessionKey] = model;
                }

                return View(model);
        }

        [HttpGet]
        public ActionResult PatientRoiPdf(int disclosureId, string lastFourOfSsn, string patientName, DateTime? dateOfBirth)
        {
            var model = new PatientRoiPdfViewModel
            {
                DateOfBirth = dateOfBirth,
                PatientLastFourOfSsn = lastFourOfSsn,
                PatientName = patientName
            };

            model.Disclosure = _roiService.GetDisclosureById(disclosureId);

            var patientRoiFilename = string.Format("{0}-{1}-{2}.pdf", patientName, lastFourOfSsn, model.Disclosure.DateTimeOfDisclosure.ToString("MM-dd-yyyy"));

            var html = this.RenderView("PatientRoiPdf", model);

            var leftFooter = string.Format("{0} DOB: {1}", patientName, dateOfBirth.HasValue ? dateOfBirth.Value.ToString("MM/dd/yyyy") : "Unknown");

            var patientRoiTempPath = _accessiblePdfService.createPdf(html, leftFooter, PdfTypes.PatientRoi);

            return File(System.IO.File.ReadAllBytes(patientRoiTempPath), "application/pdf", patientRoiFilename);
        }

        [HttpGet]
        [RoiAdminAuthorize]
        public ActionResult RoiAdminPdf()
        {
            var model = RoiAdminSearchViewModel.retrieveFromSession();

            var adminRoiFilename = string.Format("{0}-{1}-{2}-{3}-to-{4}.pdf",
                model.selectedVisnId,
                model.selectedFacilityName,
                string.IsNullOrWhiteSpace(model.selectedUserName) ? "All Users" : model.selectedUserName,
                model.StartDate.HasValue ? model.StartDate.Value.ToString("MM-dd-yyyy") : "Unknown",
                model.EndDate.HasValue ? model.EndDate.Value.ToString("MM-dd-yyyy") : "Unknown");

            var html = this.RenderView("RoiAdminPdf", model);

            var leftFooter = string.Format("VISN: {0}, Facility: {1}-{2}, User: {3}  -  Document Created: {4}",
                model.selectedVisnId, 
                model.facilityId, 
                model.selectedFacilityName, 
                string.IsNullOrWhiteSpace(model.selectedUserName) ? "All Users" : model.selectedUserName,
                DateTime.Now.ToString("MM/dd/yyyy HH:mm"));

            var RoiAdminTempPath = _accessiblePdfService.createPdf(html, leftFooter, PdfTypes.RoiAdmin, false);

            return File(System.IO.File.ReadAllBytes(RoiAdminTempPath), "application/pdf", adminRoiFilename);
        }

        [HttpGet]
        [RoiAdminAuthorize]
        public ActionResult Admin()
        {
            var roiAdmin = new RoiAdminSearchViewModel();
            roiAdmin.visnList = _userService.GetVisn().ToList();
            roiAdmin.facilityList = new List<Facility>();
            roiAdmin.UserList = new List<UserByFacility>();

            return View(roiAdmin);
        }

        [HttpGet]
        public ActionResult GetFacilitiesById(int id)
        {
            var roiAdmin = new RoiAdminSearchViewModel();
            roiAdmin.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == id).ToList();
            return Json(roiAdmin.facilityList, JsonRequestBehavior.AllowGet);
        }

        [HttpPost]
        public ActionResult GetFacilitiesById(RoiAdminSearchViewModel model)
        {
            int selectedVisnId = model.selectedVisnId;
            model.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == selectedVisnId).ToList();
            return View(model);
        }


        [HttpGet]
        public ActionResult GetUserNameByStation(int? id)
        {
            var nonNullId = id ?? 0;
            var roiAdmin = new RoiAdminSearchViewModel();
           
            roiAdmin.UserList = _roiService.GetUsersByFacilityId(nonNullId);

            return Json(roiAdmin.UserList, JsonRequestBehavior.AllowGet);
        }

        [HttpPost]
        public ActionResult GetUserNameByStation(RoiAdminSearchViewModel model)
        {
            int selectedfacilityId = 0;

            if (model.facilityId > 0 )
            {
                selectedfacilityId = model.facilityId;
            }
            
            model.UserList = _roiService.GetUsersByFacilityId(selectedfacilityId);
            return View(model);
        }

        [ValidateAntiForgeryToken]
        [HttpPost]
        public ActionResult EditDisclosureMadeTo(Disclosure disclosure)
        {
            ViewBag.UserIsRoiAdmin = _userService.CheckRoiAdminAuthorization() > 0;
            if (ModelState.IsValidField("ProviderNameOrGroup") 
                && ModelState.IsValidField("ProviderStreetAddress1")
                && ModelState.IsValidField("ProviderStreetAddress2")
                && ModelState.IsValidField("ProviderCity")
                && ModelState.IsValidField("ProviderState")
                && ModelState.IsValidField("ProviderZip")
                && ModelState.IsValidField("ProviderPhone"))
            {
                try
                {
                    var userId = System.Web.HttpContext.Current.User.Identity.Name;
                    disclosure = _roiService.EditDisclosureMadeTo(disclosure, ViewBag.UserIsRoiAdmin, userId);
                }
                catch (Exception e)
                {
                    ModelState.AddModelError(string.Empty, e.Message);
                }
                return PartialView(disclosure);
            }
            else
            {
                return PartialView(disclosure);
            }

        }

        [HttpGet]
        public ActionResult Unauthorized()
        {
            return View();
        }

    }
}